import React, { useEffect, useState } from "react";
import Link from 'next/link'; 
import Head from 'next/head'; 

// Components
import StudentDashboardSidebar from "@/components/Student/StudentDashboardSidebar";

// Utils 
import { userService } from "@/services/user.service";
import { fetchWrapper } from "@/helpers/fetch-wrapper";

import Swal from 'sweetalert2'; 

import { toast } from 'react-toastify';
import 'react-toastify/dist/ReactToastify.css';

const StudentReviews = () => {
     const [reviews, setReviews] = useState([]); 

     const fetchUserReviews = async () => {
          try {
               const response = await fetchWrapper.get('/api/users/student_reviews'); 
               setReviews(response.reviews);
          } catch (error) {
               console.error("Error fetching user reviews:", error);
          }
     };

     useEffect(() => {
          const user = userService.userValue;

		if (user) { 
               fetchUserReviews(); 
          }

	}, []);

     const handleRemoveReview = async (reviewID) => {
          const result = await Swal.fire({
               title: 'Esti sigur ca vrei sa stergi aceasta recenzie ?',
               icon: 'warning',
               showCancelButton: true,
               confirmButtonColor: '#3085d6',
               cancelButtonColor: '#d33',
               confirmButtonText: 'Da',
               cancelButtonText: 'Nu',
               customClass: {
                    popup: 'swal2-large'
               }
          });
  
          if (result.isConfirmed) {
               try {
                    await fetchWrapper.delete(`/api/users/student_reviews`, { reviewID });
                    toast.success('Recenzia a fost eliminata cu succes.');

                    fetchUserReviews(); 
               } catch (error) {
                    Swal.fire('Error!', 'A aparut o eroare. Incearca din nou.', 'error');
               }
          }
     };

     const handleEditReview = async (review) => {
          const { value: reviewText } = await Swal.fire({
               title: '<i className="feather-edit"/> Editare Recenzie',
               html: `
                    <br>
                    <p>Recenzie actuala:</p>
                    <p>${review.reviewText}</p>
                    <br/>
                    <textarea id="swal-input" class="swal2-input" placeholder="Introdu textul nou">${review.reviewText}</textarea>
               `,
               focusConfirm: false,
               preConfirm: () => {
                    const text = document.getElementById('swal-input').value;
                    if (!text) {
                         Swal.showValidationMessage('Textul nu poate fi gol');
                    }
                    return text;
               },
               confirmButtonColor: '#3085d6',
               cancelButtonColor: '#d33',
               confirmButtonText: 'Salveaza',
               showCancelButton: true,
               cancelButtonText: 'Anuleaza',
               customClass: {
                    popup: 'swal2-large'
               }
          });

          if (reviewText) {
               try {
                    await fetchWrapper.post(`/api/users/student_reviews`, { reviewID: review.id, reviewText });
                    fetchUserReviews(); 
                    Swal.fire(
                         'Salvat!',
                         'Recenzia a fost actualizata.',
                         'success'
                    );
               } catch (error) {
                    console.error("Error updating review:", error);
                    Swal.fire(
                         'Eroare!',
                         'Nu s-a putut actualiza recenzia.',
                         'error'
                    );
               }
          }
     };

	return (
		<>
               <Head>
                    <meta name="robots" content="noindex, nofollow"/>  
                    <title>Profilul meu - Recenzii</title>
               </Head>

			<div className='rbt-page-banner-wrapper'>
				<div className='rbt-banner-image' />
			</div>
			<div className='rbt-dashboard-area rbt-section-overlayping-top rbt-section-gapBottom'>
				<div className='container'>
					<div className='row'>
						<div className='col-lg-12'>
							<div className='row g-5'>
								<div className='col-lg-3'>
									<StudentDashboardSidebar />
								</div>

								<div className='col-lg-9'>
									<div className='rbt-dashboard-content bg-color-white rbt-shadow-box'>
										<div className='content'>
											<div className='section-title'>
												<h4 className='rbt-title-style-3'>
													<i className='feather-star'></i>{" "}
													Recenzii
												</h4>
											</div>

											<div className='rbt-dashboard-table table-responsive mobile-table-750'>
												<table className='rbt-table table table-borderless'>
													<thead>
														<tr>
															<th>
																Curs
															</th>
															<th>
																Recenzie oferita
															</th>
															<th></th>
														</tr>
													</thead>
													<tbody>
                                                                      {reviews.map((review, index) => (
														<tr key={index}>
															<th>
																<span className='b3'>
																	<Link href={'/' + review.courseURL}>
																		{review.courseName}
																	</Link>
																</span>
															</th>
															<td>
																<div className='rbt-review'>
																	<div className='rating'>
                                                                                          {Array.from({ length: review.reviewStars }, (_, index) => (
                                                                                               <i key={index} className="fas fa-star" style={{ color: '#FF9747' }}></i>
                                                                                          ))}

																	</div>
																</div>
																<p className='b2'>
																	{review.reviewText}
																</p>
															</td>
															<td>
																<div className='rbt-button-group justify-content-end'>
																	<Link
																		className='rbt-btn btn-xs bg-primary-opacity radius-round'
																		href='#'
																		title='Edit'
                                                                                          onClick={() => handleEditReview(review)}
																	>
																		<i className='feather-edit pl--0'></i>
																	</Link>
																	<Link
																		className='rbt-btn btn-xs bg-color-danger-opacity radius-round color-danger'
																		href='#'
																		title='Delete'
                                                                                          onClick={() => handleRemoveReview(review.id)}
																	>
																		<i className='feather-trash-2 pl--0'></i>
																	</Link>
																</div>
															</td>
														</tr>
                                                                      ))}
													</tbody>
												</table>
											</div>
										</div>
									</div>
								</div>
							</div>
						</div>
					</div>
				</div>
			</div>
		</>
	);
};

export default StudentReviews;
