import React, { useState, useEffect } from "react";
import { useRouter } from 'next/router'; 

// Next Components 
import Link from "next/link";
import Image from 'next/image'; 

// Database & Other Utils  
import { userService } from '@/services/user.service';
import { fetchWrapper } from '@/helpers/fetch-wrapper'; 

import Swal from 'sweetalert2'; 

// Toast Notifications 
import { toast } from 'react-toastify';
import 'react-toastify/dist/ReactToastify.css';

const CosDeCumparaturi = () => {
     const user = (!userService.userValue) ? (false):(true); 
     const router = useRouter();

     const [cartProducts, setCartProducts] = useState([]); 
     const [cartTotal, setCartTotal] = useState(0); 
     const [discountCode, setDiscountCode] = useState(''); 
     const [appliedDiscount, setAppliedDiscount] = useState(0); 
     const [appliedPercentage, setAppliedPercentage] = useState(0); 
     const [appliedCode, setAppliedCode] = useState(''); 

     const fetchCartProducts = async () => {
          try {
               const response = await fetchWrapper.get(`/api/users/cart`);
               setCartProducts(response.cartProducts);
          } catch (error) {
              console.error("Failed to fetch cart products", error);
          }
     };
  
     useEffect(() => {
          if(!user) router.replace('/autentificare'); 
          else {
               fetchCartProducts();
          }
     }, [user, router]);

     useEffect(() => {
          let totalPrice = 0; 
          cartProducts.forEach(cart => {
               totalPrice += parseFloat(
                    (parseFloat(cart.productPriceReduced) > 0 ? cart.productPriceReduced : cart.productPrice) * cart.productQuantity
               );
          });
          setCartTotal(totalPrice.toFixed(2));
     }, [cartProducts]);

     const handleRemove = async (productID) => {
          const result = await Swal.fire({
               title: 'Esti sigur ca vrei sa stergi acest produs din cosul tau?',
               icon: 'warning',
               showCancelButton: true,
               confirmButtonColor: '#3085d6',
               cancelButtonColor: '#d33',
               confirmButtonText: 'Da',
               cancelButtonText: 'Nu',
               customClass: {
                    popup: 'swal2-large'
               }
          });
  
          if (result.isConfirmed) {
               try {
                    await fetchWrapper.delete(`/api/users/cart`, { productID });
                    toast.success('Produsul a fost eliminat cu succes din cosul de cumparaturi.');

                    const response = await fetchWrapper.get(`/api/users/cart`);
                    setCartProducts(response.cartProducts);
                    setDiscountCode(''); 
                    setAppliedPercentage(0); 
                    setAppliedDiscount(0); 
                    setAppliedCode(''); 
               } catch (error) {
                    Swal.fire('Error!', 'A aparut o eroare. Incearca din nou.', 'error');
               }
          }
     };

     const handleDiscountCodeChange = async (event) => { 
          const code = event.target.value;
          setDiscountCode(code);
          
          if(code.length === 10) {
               try {
                    const response = await fetchWrapper.post('/api/users/checkcode', { code });
                    
                    if(response.status === 200) {
                         toast.success(response.message); 
                         const discountInCurrency = ((response.percentageDiscount / 100) * cartTotal).toFixed(2); 
                         setAppliedDiscount(discountInCurrency); 
                         setAppliedPercentage(response.percentageDiscount); 
                         setAppliedCode(response.code); 
                    }
                    else toast.error(response.error);
               } catch (error) {
                    console.error("Failed to check discount code", error);
               }
          } 
     };

     const handleCheckout = async () => {
          const formData = {
               cartProducts: cartProducts, 
               cartTotal: (cartTotal - appliedDiscount).toFixed(2),
               couponCode: appliedCode
          };

          const response = await fetchWrapper.post('/api/users/checkout_order', formData); 

          window.location.href = response.url;  
     }

     return (
          <>
               <div className="cart_area" style={{ marginTop: 30 }}>
                    <div className="container">
                         <div className="row">
                              <div className="col-12">
                                   <form action="#">
                                        <div className="cart-table table-responsive mb--60">
                                             <table className="table">
                                                  <thead>
                                                       <tr>
                                                            <th className="pro-thumbnail">Imagine</th>
                                                            <th className="pro-title">Produs</th>
                                                            <th className="pro-price">Pret</th>
                                                            <th className="pro-quantity">Cantitate</th>
                                                            <th className="pro-subtotal">Total</th>
                                                            <th className="pro-remove">Actiune</th>
                                                       </tr>
                                                  </thead>
                                                  <tbody>
                                                            {cartProducts.length === 0 ? (
                                                                 <tr>
                                                                      <td className="text-center" colSpan="6">La momentul actual nu aveți produse adăugate în coș.</td>
                                                                 </tr>
                                                            ):('')}

                                                            {cartProducts.map((cart, index) => (
                                                                 <tr key={index}>
                                                                      <td className="pro-thumbnail">
                                                                           
                                                                           <Image
                                                                                src={cart.avatar}
                                                                                width={140}
                                                                                height={111}
                                                                                alt={cart.productName}
                                                                           />
                                                                           
                                                                      </td>
                                                                      <td className="pro-title">
                                                                           <Link href={'/' + cart.url}>
                                                                                {cart.productName}
                                                                           </Link>
                                                                      </td>
                                                                      <td className="pro-price">
                                                                           {parseFloat(cart.productPriceReduced) > 0 ? (
                                                                                <div class="rbt-price justify-content-start mt--10">
                                                                                     <span class="current-price theme-gradient">{cart.productPriceReduced} RON</span>
                                                                                     <span class="off-price">{cart.productPrice} RON</span>
                                                                                </div>
                                                                           ):(cart.productPrice + ' RON')}
                                                                      </td>
                                                                      <td className="pro-quantity">
                                                                           {cart.productQuantity}
                                                                      </td>
                                                                      <td className="pro-subtotal">
                                   
                                                                      <span>
                                                                           {(
                                                                                (parseFloat(cart.productPriceReduced) > 0
                                                                                     ? parseFloat(cart.productPriceReduced)
                                                                                     : parseFloat(cart.productPrice)) * cart.productQuantity
                                                                           ).toFixed(2)} RON
                                                                      </span>
                                                                      
                                                                      </td>
                                                                      <td className="pro-remove">
                                                                           <Link href="#">
                                                                                <i className="feather-x" onClick={() => handleRemove(cart.id)}></i>
                                                                           </Link>
                                                                      </td>
                                                                 </tr>
                                                            ))}
                        
                                                       
                                                  </tbody>
                                             </table>
                                        </div>
                                   </form>

                   
                                   <div className="col-lg-12 col-12">
                                        <div className="cart-summary">
                                             <div className="cart-summary-wrap">
                                                  <p>
                                                       Total Produse <span>{cartTotal} RON</span>
                                                  </p>
                                                  <p style={{ color: 'red' }}>
                                                       Cod de reducere: {appliedCode ? ('(' + appliedCode + ')'):('')} <span>-{appliedDiscount} RON </span>
                                                  </p>
                                                  <h2>
                                                       Total de plata{" "}
                                                       <span>{(cartTotal - appliedDiscount).toFixed(2)} RON</span>
                                                  </h2>
                                                  <br/>
                                                  <p>
                                               
                                                       <input 
                                                            type="text" 
                                                            placeholder="Cod de reducere" 
                                                            value={discountCode} 
                                                            min={10}
                                                            max={10}
                                                            maxLength={10} 
                                                            onChange={handleDiscountCodeChange}
                                                            disabled={cartProducts.length === 0 ? (true):(false)}
                                                       />
                                                       {cartProducts.length > 0 ? (
                                                       <>
                                                       <small>Daca detineti un cod de reducere il puteti introduce in caseta de mai sus.</small>
                                                       <br/>
                                                       <small style={{ color: 'orange' }}>Toate codurile de reducere sunt formate din 10 caractere.</small>
                                                       <br/>
                                                       <small style={{ color: 'red' }}>Codul se valideaza automat, in timp ce este scris, daca acesta este valid, veti fii instiintati pe parcursul completarii casutei.</small></>
                                                       ):('')}
                                                  </p>

                                                  <div className="cart-submit-btn-group">
                                                       <div className="single-button w-100">
                                                            <button
                                                                 onClick={handleCheckout}
                                                                 className="rbt-btn btn-gradient rbt-switch-btn rbt-switch-y w-100 text-center"
                                                            
                                                            >
                                                                 <span data-text="Plateste"> Plateste</span>
                                                            </button>
                                                       </div>
                                                  </div>
                                             </div>
                                        
                                        </div>
                                   </div>
                           
                              </div>
                         </div>
                    </div>
               </div>
          </>
     );
};

export default CosDeCumparaturi; 
