import { Sequelize, DataTypes } from 'sequelize';
import mysql from 'mysql2/promise';
import getConfig from 'next/config';

const { serverRuntimeConfig } = getConfig();
let sequelizeInstance = null;

export const db = {
    initialized: false,
    initialize,
    sequelize: null,
    Sequelize
};

async function initialize() {
     if (!sequelizeInstance) {
          const { host, port, user, password, database } = serverRuntimeConfig.dbConfig;
          const connection = await mysql.createConnection({ host, port, user, password });
          await connection.query(`CREATE DATABASE IF NOT EXISTS \`${database}\`;`);

          sequelizeInstance = new Sequelize(database, user, password, { dialect: 'mysql', collate: 'utf8mb4_unicode_ci' });

          db.User = userModel(sequelizeInstance);
          db.Categories = categoriesModel(sequelizeInstance); 
          db.Courses = coursesModel(sequelizeInstance); 
          db.CoursesProgress = coursesProgressModel(sequelizeInstance); 
          db.Modules = modulesModel(sequelizeInstance); 
          db.Cart = cartModel(sequelizeInstance); 
          db.UserSavedAds = favoritesModel(sequelizeInstance);
          db.Reviews = reviewsModel(sequelizeInstance); 
          db.Orders = ordersModel(sequelizeInstance); 

          await sequelizeInstance.sync({ alter: false, force: false });
          
          db.sequelize = sequelizeInstance;
          db.initialized = true;
     }
}

function userModel(sequelize) {
     const attributes = {
          email: {type: DataTypes.STRING, allowNull: false }, 
          hash: { type: DataTypes.STRING, allowNull: false },
          admin: { type: DataTypes.TINYINT(1), allowNull: true, defaultValue: '0' }, 
          accountType: {type: DataTypes.TINYINT(1), allowNull: true, defaultValue: '0' }, 
          avatar: { type: DataTypes.CHAR(255), allowNull: true, defaultValue: 'https://d3rmaimnuo5t1q.cloudfront.net/images/user.webp' }, 
          background: { type: DataTypes.CHAR(255), allowNull: true, defaultValue: 'https://d3rmaimnuo5t1q.cloudfront.net/images/bg.webp'},
          userName: { type: DataTypes.CHAR(64), allowNull: true, }, 
          surName: { type: DataTypes.CHAR(64), allowNull: true }, 
          firstName: { type: DataTypes.CHAR(64), allowNull: true },
          userURL: { type: DataTypes.STRING, allowNull: true }, 
          bio: { type: DataTypes.STRING, allowNull: true }, 
          phoneNumber: { type: DataTypes.CHAR(15), allowNull: true},
          facebook: { type: DataTypes.CHAR(64), allowNull:true }, 
          twitter: { type: DataTypes.CHAR(64), allowNull: true },
          linkedin: { type: DataTypes.CHAR(64), allowNull: true }, 
          github: { type: DataTypes.CHAR(64), allowNull: true }
     };

     const options = {
          defaultScope: {
               attributes: { exclude: ['hash'] }
          },
          scopes: {
               withHash: { attributes: {}, }
          }
     };

     return sequelize.define('users', attributes, options);
}

function categoriesModel(sequelize) {
     const attributes = {
          name: {
               type: DataTypes.CHAR(64),
               allowNull: true
          },
          url: {
               type: DataTypes.CHAR(64),
               allowNull: true
          },
          parentID: {
               type: DataTypes.TINYINT(2),
               allowNull: true,
               references: {
                    model: 'categories',
                    key: 'id'
               }
          },
          title: {
               type: DataTypes.CHAR(128),
               allowNull: true
          },
          description: {
               type: DataTypes.STRING,
               allowNull: true
          },
          keywords: {
               type: DataTypes.CHAR(255),
               allowNull: true
          }
     };
 
     const Categories = sequelize.define('categories', attributes, {
          tableName: 'categories',
          timestamps: false
     });

     Categories.hasMany(Categories, {
          as: 'children',
          foreignKey: 'parentID'
     });
 
     Categories.belongsTo(Categories, {
          as: 'parent',
          foreignKey: 'parentID'
     });

     Categories.hasMany(Categories, {
          as: 'subChildren',
          foreignKey: 'parentID'
     });
 
     Categories.belongsTo(Categories, {
          as: 'subParent',
          foreignKey: 'parentID'
     });
 
     return Categories;
}

function coursesModel(sequelize) {
     const attributes = {
          instructorID: { type: DataTypes.INTEGER, allowNull: true }, 
          instructorName: { type: DataTypes.CHAR(80), allowNull: true }, 
          title: { type: DataTypes.CHAR(80), allowNull: true }, 
          description: { type: DataTypes.CHAR(80), allowNull: true }, 
          categoryName: { type: DataTypes.CHAR(64), allowNull: true }, 
          categoryURL: { type: DataTypes.CHAR(64), allowNull: true }, 
          subcategoryName: { type: DataTypes.CHAR(64), allowNull: true }, 
          subcategoryURL: { type: DataTypes.CHAR(64), allowNull: true }, 
          parentName: { type: DataTypes.CHAR(18), allowNull: true }, 
          parentURL: { type: DataTypes.CHAR(25), allowNull: true }, 
          price: { type: DataTypes.CHAR(32), allowNull: true }, 
          priceReduced: { type: DataTypes.CHAR(32), allowNull: true }, 
          previewImage: { type: DataTypes.CHAR(164), allowNull: true },
          previewVideo: { type: DataTypes.CHAR(164), allowNull: true },
          url: { type: DataTypes.CHAR(84), allowNull: true },
          level: { type: DataTypes.CHAR(10), allowNull: true },
          waiting: { type: DataTypes.TINYINT(1), allowNull: true, defaultValue: 1}
     };

     return sequelize.define('courses', attributes, {
          tableName: 'courses',
          timestamps: true 
     }); 
}

function coursesProgressModel(sequelize) {
     const attributes = {
          userID: { type: DataTypes.INTEGER, allowNull: true }, 
          courseID: { type: DataTypes.INTEGER, allowNull: true }, 
          courseName: { type: DataTypes.CHAR(80), allowNull: true }, 
          courseProgress: { type: DataTypes.TEXT('long'), allowNull: true },
          quizProgress: { type: DataTypes.TEXT('long'), allowNull: true }
     };

     return sequelize.define('courses_progress', attributes, {
          tableName: 'courses_progress',
          timestamps: false
     }); 
}

function modulesModel(sequelize) {
     const attributes = {
          courseID: { type: DataTypes.INTEGER, allowNull: false }, 
          title: { type: DataTypes.CHAR(64), allowNull: true }, 
          type: { type: DataTypes.CHAR(12), allowNull: true }, 
          videoUrl: { type: DataTypes.CHAR(164), allowNull: true },
          duration: { type: DataTypes.INTEGER, allowNull: true }, 
          parentID: { type: DataTypes.CHAR(164), allowNull: true },
          answers: { type: DataTypes.TEXT('long'), allowNull: true } 
     };

     return sequelize.define('modules', attributes, {
          tableName: 'modules',
          timestamps: false 
     }); 
}

function cartModel(sequelize) {    
     const attributes = {
          forID: { type: DataTypes.INTEGER, allowNull: true }, 
          productID: { type: DataTypes.INTEGER, allowNull: true }, 
          restaurantID: { type: DataTypes.INTEGER, allowNull: true }, 
          productName: { type: DataTypes.CHAR, allowNull: true },
          productQuantity: { type: DataTypes.TINYINT, allowNull: true, defaultValue:'1' },
          productPrice: { type: DataTypes.DECIMAL(5,2), allowNull: true }, 
          productPriceReduced: { type: DataTypes.DECIMAL(5,2), allowNull: true }, 
     }

     return sequelize.define('cart', attributes, {
          tableName: 'cart',
          timestamps: false,
     });
} 

function favoritesModel(sequelize) {
     const attributes = {
          UserID: { type: DataTypes.INTEGER, allowNull: false },
          SavedAd: { type: DataTypes.INTEGER, allowNull: false }
     }

     return sequelize.define('favorites', attributes, {
          tableName: 'favorites',
          timestamps:false 
     }); 
}

function reviewsModel(sequelize) {
     const attributes = {
          reviewBy: { type: DataTypes.STRING, allowNull: true },
          reviewByID: { type: DataTypes.INTEGER, allowNull: true }, 
          courseID: { type: DataTypes.INTEGER, allowNull: true }, 
          reviewStars: { type: DataTypes.TINYINT(1), allowNull: true }, 
          reviewText: { type: DataTypes.STRING, allowNull: false },
     }

     return sequelize.define('reviews', attributes, {
          tableName: 'reviews',
          timestamps: true,
     });
}

function ordersModel(sequelize) {
     const attributes = {
          userID: { type: DataTypes.INTEGER, allowNull: true }, 
          courseName: { type: DataTypes.CHAR(64), allowNull: true }, 
          price: { type: DataTypes.CHAR(32), allowNull: true } 
     }

     return sequelize.define('orders', attributes, {
          tableName: 'orders', 
          timestamps: true 
     });
}